function setBvAux(m)
%SETBVAUX Set auxiliary states for a greenhouse crop model based on Vanthoor (2011), with the addition of lamps and growpipes
%
% Based on the electronic appendices (the case of a Dutch greenhouse) of:
%   [1] Vanthoor, B., Stanghellini, C., van Henten, E. J. & de Visser, P. H. B. 
%       A methodology for model-based greenhouse design: Part 1, a greenhouse climate 
%       model for a broad range of designs and climates. Biosyst. Eng. 110, 363377 (2011).
%   [2] Vanthoor, B., de Visser, P. H. B., Stanghellini, C. & van Henten, E. J. 
%       A methodology for model-based greenhouse design: Part 2, description and 
%       validation of a tomato yield model. Biosyst. Eng. 110, 378395 (2011).
% These are also available as Chapters 8 and 9, respecitvely, of
%   [3] Vanthoor, B. A model based greenhouse design method. (Wageningen University, 2011).
% Other sources are:
%   [4] De Zwart, H. F. Analyzing energy-saving options in greenhouse cultivation 
%       using a simulation model. (Landbouwuniversiteit Wageningen, 1996).
% Inputs:
%   m    - a StateSpaceModel object to be used as a Vanthoor model.
%
% Everything in this file is from the references above, which can be used
% to follow the setup of the auxiliary states. In addition, toplights,
% interlights, and grow pipes have been included. The choice of parameters
% set in setBvParams influences which of these are actually active, if at
% all. To see the model extensions applied here, simply search for 'lamp/Lamp', 
% 'intLamp/IntLamp', 'gro/Gro'.

% David Katzin, Wageningen University
% david.katzin@wur.nl
    
    % shorthands for easier reading
    x = m.x;
    p = m.p;
    u = m.u;
    d = m.d;    
    
    
    %% lumped cover layers  - Section 3 [1]
    % shadow screen and permanent shadow screen
    
    % PAR transmission coefficient of the shadow screen layer [-]
    addAux(m, 'tauShScrPar', 1-u.shScr*(1-p.tauShScrPar));
    
    % PAR transmission coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'tauShScrPerPar', 1-u.shScrPer*(1-p.tauShScrPerPar));
    
    % PAR reflection coefficient of the shadow screen layer [-]
    addAux(m, 'rhoShScrPar', u.shScr*p.rhoShScrPar);
    
    % PAR reflection coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'rhoShScrPerPar', u.shScrPer*p.rhoShScrPerPar);
        
    % PAR transmission coefficient of the shadow screen and semi permanent shadow screen layer [-]
    % Equation 16 [1]
    addAux(m, 'tauShScrShScrPerPar', tau12(m.a.tauShScrPar, m.a.tauShScrPerPar, ...
        m.a.rhoShScrPar, m.a.rhoShScrPar, m.a.rhoShScrPerPar, m.a.rhoShScrPerPar));
    
    % PAR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the top [-]
    % Equation 17 [1]
    addAux(m, 'rhoShScrShScrPerParUp', rhoUp(m.a.tauShScrPar, m.a.tauShScrPerPar, ...
        m.a.rhoShScrPar, m.a.rhoShScrPar, m.a.rhoShScrPerPar, m.a.rhoShScrPerPar));
    
    % PAR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the bottom [-]
    % Equation 17 [1]
    addAux(m, 'rhoShScrShScrPerParDn', rhoDn(m.a.tauShScrPar, m.a.tauShScrPerPar, ...
        m.a.rhoShScrPar, m.a.rhoShScrPar, m.a.rhoShScrPerPar, m.a.rhoShScrPerPar));
    
    % NIR transmission coefficient of the shadow screen layer [-]
    addAux(m, 'tauShScrNir', 1-u.shScr*(1-p.tauShScrNir));
    
    % NIR transmission coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'tauShScrPerNir', 1-u.shScrPer*(1-p.tauShScrPerNir));
    
    % NIR reflection coefficient of the shadow screen layer [-]
    addAux(m, 'rhoShScrNir', u.shScr*p.rhoShScrNir);
    
    % NIR reflection coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'rhoShScrPerNir', u.shScrPer*p.rhoShScrPerNir);
        
    % NIR transmission coefficient of the shadow screen and semi permanent shadow screen layer [-]
    addAux(m, 'tauShScrShScrPerNir', tau12(m.a.tauShScrNir, m.a.tauShScrPerNir, ...
        m.a.rhoShScrNir, m.a.rhoShScrNir, m.a.rhoShScrPerNir, m.a.rhoShScrPerNir));
    
    % NIR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the top [-]
    addAux(m, 'rhoShScrShScrPerNirUp', rhoUp(m.a.tauShScrNir, m.a.tauShScrPerNir, ...
        m.a.rhoShScrNir, m.a.rhoShScrNir, m.a.rhoShScrPerNir, m.a.rhoShScrPerNir));
    
    % NIR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the bottom [-]
    addAux(m, 'rhoShScrShScrPerNirDn', rhoDn(m.a.tauShScrNir, m.a.tauShScrPerNir, ...
        m.a.rhoShScrNir, m.a.rhoShScrNir, m.a.rhoShScrPerNir, m.a.rhoShScrPerNir));
    
    % FIR  transmission coefficient of the shadow screen layer [-]
    addAux(m, 'tauShScrFir', 1-u.shScr*(1-p.tauShScrFir));
    
    % FIR transmission coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'tauShScrPerFir', 1-u.shScrPer*(1-p.tauShScrPerFir));
    
    % FIR reflection coefficient of the shadow screen layer [-]
    addAux(m, 'rhoShScrFir', u.shScr*p.rhoShScrFir);
    
    % FIR reflection coefficient of the semi-permanent shadow screen layer [-]
    addAux(m, 'rhoShScrPerFir', u.shScrPer*p.rhoShScrPerFir);
        
    % FIR transmission coefficient of the shadow screen and semi permanent shadow screen layer [-]
    addAux(m, 'tauShScrShScrPerFir', tau12(m.a.tauShScrFir, m.a.tauShScrPerFir, ...
        m.a.rhoShScrFir, m.a.rhoShScrFir, m.a.rhoShScrPerFir, m.a.rhoShScrPerFir));
    
    % FIR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the top [-]
    addAux(m, 'rhoShScrShScrPerFirUp', rhoUp(m.a.tauShScrFir, m.a.tauShScrPerFir, ...
        m.a.rhoShScrFir, m.a.rhoShScrFir, m.a.rhoShScrPerFir, m.a.rhoShScrPerFir));
    
    % FIR reflection coefficient of the shadow screen and semi permanent shadow screen layer towards the bottom [-]
    addAux(m, 'rhoShScrShScrPerFirDn', rhoDn(m.a.tauShScrFir, m.a.tauShScrPerFir, ...
        m.a.rhoShScrFir, m.a.rhoShScrFir, m.a.rhoShScrPerFir, m.a.rhoShScrPerFir));
    
    % thermal screen and roof
    % PAR

    % PAR transmission coefficient of the thermal screen [-]
    addAux(m, 'tauThScrPar', 1-u.thScr*(1-p.tauThScrPar));
    
    % PAR reflection coefficient of the thermal screen [-]
    addAux(m, 'rhoThScrPar', u.thScr*p.rhoThScrPar);
    
    % PAR transmission coefficient of the thermal screen and roof [-]
    addAux(m, 'tauCovThScrPar', tau12(p.tauRfPar, m.a.tauThScrPar, ...
        p.rhoRfPar, p.rhoRfPar, m.a.rhoThScrPar, m.a.rhoThScrPar));
    
    % PAR reflection coefficient of the thermal screen and roof towards the top [-]
    addAux(m, 'rhoCovThScrParUp', rhoUp(p.tauRfPar, m.a.tauThScrPar, ...
        p.rhoRfPar, p.rhoRfPar, m.a.rhoThScrPar, m.a.rhoThScrPar));
    
    % PAR reflection coefficient of the thermal screen and roof towards the bottom [-]
    addAux(m, 'rhoCovThScrParDn', rhoDn(p.tauRfPar, m.a.tauThScrPar, ...
        p.rhoRfPar, p.rhoRfPar, m.a.rhoThScrPar, m.a.rhoThScrPar));
   
    % PAR transmission coefficient of the blackout screen [-]
    addAux(m, 'tauBlScrPar', 1-u.blScr*(1-p.tauBlScrPar));
    
    % PAR reflection coefficient of the blackout screen [-]
    addAux(m, 'rhoBlScrPar', u.blScr*p.rhoBlScrPar);
    
    % PAR transmission coefficient of the blackout screen, thermal screen and roof [-]
    addAux(m, 'tauCovScrPar', tau12(m.a.tauCovThScrPar, m.a.tauBlScrPar, ...
        m.a.rhoCovThScrParUp, m.a.rhoCovThScrParDn, ...
        m.a.rhoBlScrPar, m.a.rhoBlScrPar));
    
    % PAR reflection coefficient of the blackout screen, thermal screen and roof towards the top [-]
    addAux(m, 'rhoCovScrParUp', rhoUp(m.a.tauCovThScrPar, m.a.tauBlScrPar, ...
        m.a.rhoCovThScrParUp, m.a.rhoCovThScrParDn, ...
        m.a.rhoBlScrPar, m.a.rhoBlScrPar));
    
    % PAR reflection coefficient of the blackout screen, thermal screen and roof towards the bottom [-]
    addAux(m, 'rhoCovScrParDn', rhoDn(m.a.tauCovThScrPar, m.a.tauBlScrPar, ...
        m.a.rhoCovThScrParUp, m.a.rhoCovThScrParDn, ...
        m.a.rhoBlScrPar, m.a.rhoBlScrPar));
    
    % NIR
    % NIR transmission coefficient of the thermal screen [-]
    addAux(m, 'tauThScrNir', 1-u.thScr*(1-p.tauThScrNir));
    
    % NIR reflection coefficient of the thermal screen [-]
    addAux(m, 'rhoThScrNir', u.thScr*p.rhoThScrNir);
    
    % NIR transmission coefficient of the thermal screen and roof [-]
    addAux(m, 'tauCovThScrNir', tau12(p.tauRfNir, m.a.tauThScrNir, ...
        p.rhoRfNir, p.rhoRfNir, m.a.rhoThScrNir, m.a.rhoThScrNir));
    
    % NIR reflection coefficient of the thermal screen and roof towards the top [-]
    addAux(m, 'rhoCovThScrNirUp', rhoUp(p.tauRfNir, m.a.tauThScrNir, ...
        p.rhoRfNir, p.rhoRfNir, m.a.rhoThScrNir, m.a.rhoThScrNir));
    
    % NIR reflection coefficient of the thermal screen and roof towards the top [-]
    addAux(m, 'rhoCovThScrNirDn', rhoDn(p.tauRfNir, m.a.tauThScrNir, ...
        p.rhoRfNir, p.rhoRfNir, m.a.rhoThScrNir, m.a.rhoThScrNir));
    
    % NIR transmission coefficient of the blackout screen [-]
    addAux(m, 'tauBlScrNir', 1-u.blScr*(1-p.tauBlScrNir));
    
    % NIR reflection coefficient of the blackout screen [-]
    addAux(m, 'rhoBlScrNir', u.blScr*p.rhoBlScrNir);
    
    % NIR transmission coefficient of the blackout screen, thermal screen and roof [-]
    addAux(m, 'tauCovScrNir', tau12(m.a.tauCovThScrNir, m.a.tauBlScrNir, ...
        m.a.rhoCovThScrNirUp, m.a.rhoCovThScrNirDn, ...
        m.a.rhoBlScrNir, m.a.rhoBlScrNir));
    
    % NIR reflection coefficient of the blackout screen, thermal screen and roof towards the top [-]
    addAux(m, 'rhoCovScrNirUp', rhoUp(m.a.tauCovThScrNir, m.a.tauBlScrNir, ...
        m.a.rhoCovThScrNirUp, m.a.rhoCovThScrNirDn, ...
        m.a.rhoBlScrNir, m.a.rhoBlScrNir));
    
    % NIR reflection coefficient of the blackout screen, thermal screen and roof towards the bottom [-]
    addAux(m, 'rhoCovScrNirDn', rhoDn(m.a.tauCovThScrNir, m.a.tauBlScrNir, ...
        m.a.rhoCovThScrNirUp, m.a.rhoCovThScrNirDn, ...
        m.a.rhoBlScrNir, m.a.rhoBlScrNir));
    
    % FIR
    % FIR transmission coefficient of the thermal screen [-]
    addAux(m, 'tauThScrFir', 1-u.thScr*(1-p.tauThScrFir));
    
    % FIR reflection coefficient of the thermal screen [-]
    addAux(m, 'rhoThScrFir', u.thScr*p.rhoThScrFir);
    
    % all layers, excluding lamps
    % PAR transmission coefficient of the cover [-]
    addAux(m, 'tauCovParAboveLamp', tau12(m.a.tauShScrShScrPerPar, m.a.tauCovScrPar, ...
        m.a.rhoShScrShScrPerParUp, m.a.rhoShScrShScrPerParDn, ...
        m.a.rhoCovScrParUp, m.a.rhoCovScrParDn));
    
    % PAR reflection coefficient of the cover towards the top [-]
    addAux(m, 'rhoCovParAboveLampUp', rhoUp(m.a.tauShScrShScrPerPar, m.a.tauCovScrPar, ...
        m.a.rhoShScrShScrPerParUp, m.a.rhoShScrShScrPerParDn, ...
        m.a.rhoCovScrParUp, m.a.rhoCovScrParDn));
    
    % PAR reflection coefficient of the cover towards the bottom [-]
    addAux(m, 'rhoCovParAboveLampDn', rhoDn(m.a.tauShScrShScrPerPar, m.a.tauCovScrPar, ...
        m.a.rhoShScrShScrPerParUp, m.a.rhoShScrShScrPerParDn, ...
        m.a.rhoCovScrParUp, m.a.rhoCovScrParDn));
    
    % NIR transmission coefficient of the cover [-]
    addAux(m, 'tauCovNirAboveLamp', tau12(m.a.tauShScrShScrPerNir, m.a.tauCovScrNir, ...
        m.a.rhoShScrShScrPerNirUp, m.a.rhoShScrShScrPerNirDn, ...
        m.a.rhoCovScrNirUp, m.a.rhoCovScrNirDn));
    
    % NIR reflection coefficient of the cover towards the top [-]
    addAux(m, 'rhoCovNirAboveLampUp', rhoUp(m.a.tauShScrShScrPerNir, m.a.tauCovScrNir, ...
        m.a.rhoShScrShScrPerNirUp, m.a.rhoShScrShScrPerNirDn, ...
        m.a.rhoCovScrNirUp, m.a.rhoCovScrNirDn));
    
    % NIR reflection coefficient of the cover towards the bottom [-]
    addAux(m, 'rhoCovNirAboveLampDn', rhoDn(m.a.tauShScrShScrPerNir, m.a.tauCovScrNir, ...
        m.a.rhoShScrShScrPerNirUp, m.a.rhoShScrShScrPerNirDn, ...
        m.a.rhoCovScrNirUp, m.a.rhoCovScrNirDn));
    
    % FIR transmission coefficient of the cover, excluding thermal screen and lamps [-]
    addAux(m, 'tauCovFir', tau12(m.a.tauShScrShScrPerFir, p.tauRfFir, ...
        m.a.rhoShScrShScrPerFirUp, m.a.rhoShScrShScrPerFirDn, ...
        p.rhoRfFir, p.rhoRfFir));
    
    % FIR reflection coefficient of the cover, excluding thermal screen and lamps [-]
    addAux(m, 'rhoCovFir', rhoUp(m.a.tauShScrShScrPerFir, p.tauRfFir, ...
        m.a.rhoShScrShScrPerFirUp, m.a.rhoShScrShScrPerFirDn, ...
        p.rhoRfFir, p.rhoRfFir));   
    
    % FIR transmission coefficient of the blackout screen [-]
    addAux(m, 'tauBlScrFir', 1-u.blScr*(1-p.tauBlScrFir));
    
    % FIR reflection coefficient of the blackout screen [-]
    addAux(m, 'rhoBlScrFir', u.blScr*p.rhoBlScrFir);
    
	% all layers, including lamps	
    % PAR transmission coefficient of the cover [-]
    addAux(m, 'tauCovPar', tau12(m.a.tauCovParAboveLamp, p.tauLampPar, ...
        m.a.rhoCovParAboveLampUp, m.a.rhoCovParAboveLampDn, ...
        p.rhoLampPar, p.rhoLampPar));
    
    % PAR reflection coefficient of the cover [-]
    addAux(m, 'rhoCovPar', rhoUp(m.a.tauCovParAboveLamp, p.tauLampPar, ...
        m.a.rhoCovParAboveLampUp, m.a.rhoCovParAboveLampDn, ...
        p.rhoLampPar, p.rhoLampPar));
    
    % NIR transmission coefficient of the cover [-]
    addAux(m, 'tauCovNir', tau12(m.a.tauCovNirAboveLamp, p.tauLampNir, ...
        m.a.rhoCovNirAboveLampUp, m.a.rhoCovNirAboveLampDn, ...
        p.rhoLampNir, p.rhoLampNir));
    
    % NIR reflection coefficient of the cover [-]
    addAux(m, 'rhoCovNir', rhoUp(m.a.tauCovNirAboveLamp, p.tauLampNir, ...
        m.a.rhoCovNirAboveLampUp, m.a.rhoCovNirAboveLampDn, ...
        p.rhoLampNir, p.rhoLampNir));
    
    % PAR absorption coefficient of the cover [-]
    addAux(m, 'aCovPar', 1-m.a.tauCovPar-m.a.rhoCovPar);
    
    % NIR absorption coefficient of the cover [-]
    addAux(m, 'aCovNir', 1-m.a.tauCovNir-m.a.rhoCovNir);
    
    % FIR absorption coefficient of the cover [-]
    addAux(m, 'aCovFir', 1-m.a.tauCovFir-m.a.rhoCovFir);
    
    % FIR emission coefficient of the cover [-]
    % See comment before equation 18 [1]
    addAux(m, 'epsCovFir', m.a.aCovFir);
    
    % Heat capacity of the lumped cover [J K^{-1} m^{-2}]
    % Equation 18 [1]
    addAux(m, 'capCov', cosd(p.psi)*(u.shScrPer*p.hShScrPer*p.rhoShScrPer*p.cPShScrPer+ ...
        p.hRf*p.rhoRf*p.cPRf));
    
	%% Capacities - Section 4 [1]
    
    % Leaf area index [m^2{leaf} m^{-2}]
    % Equation 5 [2]
    addAux(m, 'lai', p.sla*x.cLeaf); 
    
    % Heat capacity of canopy [J K^{-1} m^{-2}]
    % Equation 19 [1]
    addAux(m, 'capCan', p.capLeaf*m.a.lai);
    
    % Heat capacity of external and internal cover [J K^{-1} m^{-2}]
    % Equation 20 [1]
    addAux(m, 'capCovE', 0.1*m.a.capCov);
    addAux(m, 'capCovIn', 0.1*m.a.capCov);
    
    % Vapor capacity of main compartment [kg m J^{-1}] 
    % Equation 24 [1]
    addAux(m, 'capVpAir', p.mWater*p.hAir./(p.R*(x.tAir+273.15)));
    
    % Vapor capacity of top compartment [kg m J^{-1}] 
    addAux(m, 'capVpTop', p.mWater*(p.hGh-p.hAir)./(p.R*(x.tTop+273.15)));
    
    %% Global, PAR, and NIR heat fluxes - Section 5.1 [1]
    
    % Lamp electrical input [W m^{-2}]
    addAux(m, 'lampIn', p.lampsMax*u.lamp);
    
    % Interlight electrical input [W m^{-2}]
    addAux(m, 'intLampIn', p.intLampsMax*u.intLamp);
    
    % PAR above the canopy from the sun [W m^{-2}]
    % Equation 27 [1]
    addAux(m, 'rParGhSun', (1-p.etaGlobAir).*m.a.tauCovPar.*p.etaGlobPar.*d.iGlob); 
    
    % PAR above the canopy from the lamps [W m^{-2}] 
    addAux(m, 'rParGhLamp', p.etaLampPar*m.a.lampIn);
    
    % Global radiation above the canopy from the sun [W m^{-2}]
    % (PAR+NIR, where UV is counted together with NIR)
    addAux(m, 'rCanSun', (1-p.etaGlobAir).*d.iGlob.*...
        (p.etaGlobPar*m.a.tauCovPar+p.etaGlobNir*m.a.tauCovNir));
                                    % perhaps tauHatCovNir should be used here?
    
    % Global radiation above the canopy from the lamps [W m^{-2}]
    % (PAR+NIR, where UV is counted together with NIR)
    addAux(m, 'rCanLamp', (p.etaLampPar+p.etaLampNir)*m.a.lampIn);    
    
    % Global radiation to the canopy from the interlight lamps [W m^{-2}]
    % (PAR+NIR, where UV is counted together with NIR)
    addAux(m, 'rCanIntLamp', (p.etaIntLampPar+p.etaIntLampNir)*m.a.intLampIn);    
    
    % Global radiation above the canopy [W m^{-2}]
    % (PAR+NIR, where UV is counted together with NIR)
    addAux(m, 'rCan', m.a.rCanSun+m.a.rCanLamp+m.a.rCanIntLamp);    
    
    % PAR from the sun directly absorbed by the canopy [W m^{-2}]
    % Equation 26 [1]
    addAux(m, 'rParCanDownSun', m.a.rParGhSun.*(1-p.rhoCanPar).*(1-exp(-p.k1Par*m.a.lai)));
    
    % PAR from the lamps directly absorbed by the canopy [W m^{-2}]
    addAux(m, 'rParCanDownLamp', m.a.rParGhLamp.*(1-p.rhoCanPar).*(1-exp(-p.k1Par*m.a.lai)));
    
    % PAR from the sun absorbed by the canopy after reflection from the floor [W m^{-2}]
    % Equation 28 [1]
    addAux(m, 'rParFlrCanUpSun', mulNoBracks(m.a.rParGhSun,exp(-p.k1Par*m.a.lai)*p.rhoFlrPar* ...
        (1-p.rhoCanPar).*(1-exp(-p.k2Par*m.a.lai))));
    
    % PAR from the lamps absorbed by the canopy after reflection from the floor [W m^{-2}]
    addAux(m, 'rParFlrCanUpLamp', m.a.rParGhLamp.*exp(-p.k1Par*m.a.lai)*p.rhoFlrPar* ...
        (1-p.rhoCanPar).*(1-exp(-p.k2Par*m.a.lai)));
    
    % Total PAR from the sun absorbed by the canopy [W m^{-2}]
    % Equation 25 [1]
    addAux(m, 'rParSunCan', m.a.rParCanDownSun + m.a.rParFlrCanUpSun);
    
    % Total PAR from the lamps absorbed by the canopy [W m^{-2}]
    addAux(m, 'rParLampCan', m.a.rParCanDownLamp + m.a.rParFlrCanUpLamp);
    
    % Virtual NIR transmission for the cover-canopy-floor lumped model [-]
    % Equation 29 [1]
    addAux(m, 'tauHatCovNir', 1-m.a.rhoCovNir);
    addAux(m, 'tauHatFlrNir', 1-p.rhoFlrNir);
    
    % NIR transmission coefficient of the canopy [-]
    % Equation 30 [1]   
    addAux(m, 'tauHatCanNir', exp(-p.kNir*m.a.lai));
    
    % NIR reflection coefficient of the canopy [-]
    % Equation 31 [1]
    addAux(m, 'rhoHatCanNir', p.rhoCanNir*(1-m.a.tauHatCanNir));
    
    % NIR transmission coefficient of the cover and canopy [-]
    addAux(m, 'tauCovCanNir', tau12(m.a.tauHatCovNir, m.a.tauHatCanNir, m.a.rhoCovNir, m.a.rhoCovNir, m.a.rhoHatCanNir, m.a.rhoHatCanNir));
    
    % NIR reflection coefficient of the cover and canopy towards the top [-]
    addAux(m, 'rhoCovCanNirUp', rhoUp(m.a.tauHatCovNir, m.a.tauHatCanNir, m.a.rhoCovNir, m.a.rhoCovNir, m.a.rhoHatCanNir, m.a.rhoHatCanNir));
    
    % NIR reflection coefficient of the cover and canopy towards the bottom [-]
    addAux(m, 'rhoCovCanNirDn', rhoDn(m.a.tauHatCovNir, m.a.tauHatCanNir, m.a.rhoCovNir, m.a.rhoCovNir, m.a.rhoHatCanNir, m.a.rhoHatCanNir));
    
    % NIR transmission coefficient of the cover, canopy and floor [-]
    addAux(m, 'tauCovCanFlrNir', tau12(m.a.tauCovCanNir, m.a.tauHatFlrNir, m.a.rhoCovCanNirUp, m.a.rhoCovCanNirDn, p.rhoFlrNir, p.rhoFlrNir));
    
    % NIR reflection coefficient of the cover, canopy and floor [-]
    addAux(m, 'rhoCovCanFlrNir', rhoUp(m.a.tauCovCanNir, m.a.tauHatFlrNir, m.a.rhoCovCanNirUp, m.a.rhoCovCanNirDn, p.rhoFlrNir, p.rhoFlrNir));
    
    % The calculated absorption coefficient equals m.a.aCanNir [-]
    % pg. 23 [1]
    addAux(m, 'aCanNir', 1 - m.a.tauCovCanFlrNir - m.a.rhoCovCanFlrNir);
    
    % The calculated transmission coefficient equals m.a.aFlrNir [-]
    % pg. 23 [1]
    addAux(m, 'aFlrNir', m.a.tauCovCanFlrNir);
    
    % NIR from the sun absorbed by the canopy [W m^{-2}]
    % Equation 32 [1]
    addAux(m, 'rNirSunCan', (1-p.etaGlobAir).*m.a.aCanNir.*p.etaGlobNir.*d.iGlob);
    
    % NIR from the lamps absorbed by the canopy [W m^{-2}]
    addAux(m, 'rNirLampCan', p.etaLampNir.*m.a.lampIn.*(1-p.rhoCanNir).*(1-exp(-p.kNir*m.a.lai)));
    % perhaps a "virtual absorption coefficient" should be applied here too?
    
    % NIR from the sun absorbed by the floor [W m^{-2}]
    % Equation 33 [1]
    addAux(m, 'rNirSunFlr', (1-p.etaGlobAir).*m.a.aFlrNir.*p.etaGlobNir.*d.iGlob);
    
    % NIR from the lamps absorbed by the floor [W m^{-2}]
    addAux(m, 'rNirLampFlr', (1-p.rhoFlrNir).*exp(-p.kNir*m.a.lai).*p.etaLampNir.*m.a.lampIn);
    
    % PAR from the sun absorbed by the floor [W m^{-2}]
    % Equation 34 [1]
    addAux(m, 'rParSunFlr', (1-p.rhoFlrPar).*exp(-p.k1Par*m.a.lai).*m.a.rParGhSun);
    
    % PAR from the lamps absorbed by the floor [W m^{-2}]
    addAux(m, 'rParLampFlr', (1-p.rhoFlrPar).*exp(-p.k1Par*m.a.lai).*m.a.rParGhLamp);
    
	% PAR and NIR from the lamps absorbed by the greenhouse air [W m^{-2}]
	addAux(m, 'rLampAir', (p.etaLampPar+p.etaLampNir)*m.a.lampIn - m.a.rParLampCan - ...
		m.a.rNirLampCan - m.a.rParLampFlr - m.a.rNirLampFlr);
	
    % Global radiation from the sun absorbed by the greenhouse air [W m^{-2}]
    % Equation 35 [1]
    addAux(m, 'rGlobSunAir', p.etaGlobAir*d.iGlob.*...
        (m.a.tauCovPar*p.etaGlobPar+(m.a.aCanNir+m.a.aFlrNir)*p.etaGlobNir));
    
    % Global radiation from the sun absorbed by the cover [W m^{-2}]
    % Equation 36 [1]
    addAux(m, 'rGlobSunCovE', (m.a.aCovPar*p.etaGlobPar+m.a.aCovNir*p.etaGlobNir).*d.iGlob);
    
    % PAR from the interlights to the canopy lamps [W m^{-2}] 
    addAux(m, 'rParIntLampCan', p.etaIntLampPar*m.a.intLampIn);
    
    % NIR from the interlight absorbed by the canopy [W m^{-2}]
    addAux(m, 'rNirIntLampCan', p.etaIntLampNir.*m.a.lampIn);
    
    %% FIR heat fluxes - Section 5.2 [1]
	
	% FIR transmission coefficient of the thermal screen
	% Equation 38 [1]
    addAux(m, 'tauThScrFirU', (1-u.thScr*(1-p.tauThScrFir)));
	
    % FIR transmission coefficient of the blackout screen
    addAux(m, 'tauBlScrFirU', (1-u.blScr*(1-p.tauBlScrFir)));
    
	% Surface of canopy per floor area [-]
	% Table 3 [1]
    addAux(m, 'aCan', 1-exp(-p.kFir*m.a.lai));
	
	% FIR between greenhouse objects [W m^{-2}]
	% Table 3 [1]
	
	% FIR between canopy and cover [W m^{-2}]
    addAux(m, 'rCanCovIn', fir(m.a.aCan, p.epsCan, m.a.epsCovFir, p.tauLampFir*m.a.tauThScrFirU.*m.a.tauBlScrFirU, x.tCan, x.tCovIn, ...
         m.a.capCan, m.a.capCovIn, p.simRes));
	
	% FIR between canopy and sky [W m^{-2}]
    addAux(m, 'rCanSky', fir(m.a.aCan, p.epsCan, p.epsSky, p.tauLampFir*m.a.tauCovFir.*m.a.tauThScrFirU.*m.a.tauBlScrFirU, x.tCan, d.tSky, ...
        m.a.capCan, m.a.capCan, p.simRes));
	
	% FIR between canopy and thermal screen [W m^{-2}]
    addAux(m, 'rCanThScr', fir(m.a.aCan, p.epsCan, p.epsThScrFir, p.tauLampFir*u.thScr.*m.a.tauBlScrFirU, x.tCan, x.tThScr, ...
        m.a.capCan, p.capThScr, p.simRes));
    
	% FIR between canopy and floor [W m^{-2}]
	addAux(m, 'rCanFlr', fir(m.a.aCan, p.epsCan, p.epsFlr, p.fCanFlr, x.tCan, x.tFlr, ...
        m.a.capCan, p.capFlr, p.simRes));
	
	% FIR between pipes and cover [W m^{-2}]
    addAux(m, 'rPipeCovIn', fir(p.aPipe, p.epsPipe, m.a.epsCovFir, ...
        p.tauLampFir*m.a.tauThScrFirU.*m.a.tauBlScrFirU*0.49.*exp(-p.kFir*m.a.lai), x.tPipe, x.tCovIn, ...
        p.capPipe, m.a.capCovIn, p.simRes));
		
	% FIR between pipes and sky [W m^{-2}]
    addAux(m, 'rPipeSky', fir(p.aPipe, p.epsPipe, p.epsSky, ...
        p.tauLampFir*m.a.tauCovFir.*m.a.tauThScrFirU.*m.a.tauBlScrFirU*0.49.*exp(-p.kFir*m.a.lai), x.tPipe, d.tSky, ...
        p.capPipe, p.capPipe, p.simRes));
		
	% FIR between pipes and thermal screen [W m^{-2}]
    addAux(m, 'rPipeThScr', fir(p.aPipe, p.epsPipe, p.epsThScrFir, ...
        p.tauLampFir*u.thScr.*m.a.tauBlScrFirU*0.49.*exp(-p.kFir*m.a.lai), x.tPipe, x.tThScr, ...
        p.capPipe, p.capThScr, p.simRes));
		
	% FIR between pipes and floor [W m^{-2}]
    addAux(m, 'rPipeFlr', fir(p.aPipe, p.epsPipe, p.epsFlr, 0.49, x.tPipe, x.tFlr, ...
        p.capPipe, p.capFlr, p.simRes));
	
	% FIR between pipes and canopy [W m^{-2}]
    addAux(m, 'rPipeCan', fir(p.aPipe, p.epsPipe, p.epsCan, ...
        0.49.*(1-exp(-p.kFir*m.a.lai)), x.tPipe, x.tCan, ...
        p.capPipe, m.a.capCan, p.simRes));
		
	% FIR between floor and cover [W m^{-2}]
    addAux(m, 'rFlrCovIn', fir(1, p.epsFlr, m.a.epsCovFir, ...
        p.tauLampFir*m.a.tauThScrFirU.*m.a.tauBlScrFirU*(1-0.49*pi*p.lPipe*p.phiPipeE).*exp(-p.kFir*m.a.lai), x.tFlr, x.tCovIn, ...
        p.capFlr, m.a.capCovIn, p.simRes));
    
	% FIR between floor and sky [W m^{-2}]
	addAux(m, 'rFlrSky', fir(1, p.epsFlr, p.epsSky, ...
        p.tauLampFir*m.a.tauCovFir.*m.a.tauThScrFirU.*m.a.tauBlScrFirU*(1-0.49*pi*p.lPipe*p.phiPipeE).*exp(-p.kFir*m.a.lai), x.tFlr, d.tSky, ...
        p.capFlr, p.capFlr, p.simRes));
    
	% FIR between floor and thermal screen [W m^{-2}]
	addAux(m, 'rFlrThScr', fir(1, p.epsFlr, p.epsThScrFir, ...
        p.tauLampFir*u.thScr.*m.a.tauBlScrFirU*(1-0.49*pi*p.lPipe*p.phiPipeE).*exp(-p.kFir*m.a.lai), x.tFlr, x.tThScr, ...
        p.capFlr, p.capThScr, p.simRes));
    
	% FIR between thermal screen and cover [W m^{-2}]
	addAux(m, 'rThScrCovIn', fir(1, p.epsThScrFir, m.a.epsCovFir, u.thScr, x.tThScr, x.tCovIn, ...
        p.capThScr, m.a.capCovIn, p.simRes));
    
	% FIR between thermal screen and sky [W m^{-2}]
	addAux(m, 'rThScrSky', fir(1, p.epsThScrFir, p.epsSky, m.a.tauCovFir.*u.thScr, x.tThScr, d.tSky, ...
        p.capThScr, p.capThScr, p.simRes));
    
	% FIR between cover and sky [W m^{-2}]
	addAux(m, 'rCovESky', fir(1, m.a.aCovFir, p.epsSky, 1, x.tCovE, d.tSky, ...
        m.a.capCovE, m.a.capCovE, p.simRes));
    
    % FIR between lamps and floor [W m^{-2}]
    addAux(m, 'rFirLampFlr', fir(p.aLamp, p.epsLampBottom, p.epsFlr, ...
        (1-0.49*pi*p.lPipe*p.phiPipeE).*exp(-p.kFir*m.a.lai), x.tLamp, x.tFlr, ...
        p.capLamp, p.capFlr, p.simRes));
    
    % FIR between lamps and pipe [W m^{-2}]
    addAux(m, 'rLampPipe', fir(p.aLamp, p.epsLampBottom, p.epsPipe, ...
        0.49*pi*p.lPipe*p.phiPipeE.*exp(-p.kFir*m.a.lai), x.tLamp, x.tPipe, ...
        p.capLamp, p.capPipe, p.simRes));
    
    % FIR between lamps and canopy [W m^{-2}]
    addAux(m, 'rFirLampCan', fir(p.aLamp, p.epsLampBottom, p.epsCan, m.a.aCan, x.tLamp, x.tCan, ...
        p.capLamp, m.a.capCan, p.simRes));
    
    % FIR between lamps and thermal screen [W m^{-2}]
    addAux(m, 'rLampThScr', fir(p.aLamp, p.epsLampTop, p.epsThScrFir, u.thScr.*m.a.tauBlScrFirU, x.tLamp, x.tThScr, ...
        p.capLamp, p.capThScr, p.simRes));
    
    % FIR between lamps and cover [W m^{-2}]
    addAux(m, 'rLampCovIn', fir(p.aLamp, p.epsLampTop, m.a.epsCovFir, m.a.tauThScrFirU.*m.a.tauBlScrFirU, x.tLamp, x.tCovIn, ...
        p.capLamp, m.a.capCovIn, p.simRes));
    
    % FIR between lamps and sky [W m^{-2}]
    addAux(m, 'rLampSky', fir(p.aLamp, p.epsLampTop, p.epsSky, m.a.tauCovFir.*m.a.tauThScrFirU.*m.a.tauBlScrFirU, x.tLamp, d.tSky, ...
        p.capLamp, p.capLamp, p.simRes));
    
    % FIR between grow pipes and canopy [W m^{-2}]
    addAux(m, 'rGroPipeCan', fir(p.aGroPipe, p.epsGroPipe, p.epsCan, ...
        1, x.tGroPipe, x.tCan, p.capGroPipe, m.a.capCan, p.simRes));
    
    % FIR between interlights and canopy [W m^{-2}]
    addAux(m, 'rFirIntLampCan', fir(p.aIntLamp, p.epsIntLamp, p.epsCan, ...
        1, x.tIntLamp, x.tCan, p.capIntLamp, m.a.capCan, p.simRes));
    
    % FIR between blackout screen and floor [W m^{-2}]	
    addAux(m, 'rFlrBlScr', fir(1, p.epsFlr, p.epsBlScrFir, ...
        p.tauLampFir*u.blScr*(1-0.49*pi*p.lPipe*p.phiPipeE).*exp(-p.kFir*m.a.lai), x.tFlr, x.tBlScr, ...
        p.capFlr, p.capBlScr, p.simRes));
    
    % FIR between blackout screen and pipe [W m^{-2}]
    addAux(m, 'rPipeBlScr', fir(p.aPipe, p.epsPipe, p.epsBlScrFir, ...
        p.tauLampFir*u.blScr*0.49.*exp(-p.kFir*m.a.lai), x.tPipe, x.tBlScr, ...
        p.capPipe, p.capBlScr, p.simRes));
    
    % FIR between blackout screen and canopy [W m^{-2}]
    addAux(m, 'rCanBlScr', fir(m.a.aCan, p.epsCan, p.epsBlScrFir, p.tauLampFir*u.blScr, x.tCan, x.tBlScr, ...
        m.a.capCan, p.capBlScr, p.simRes));
    
    % FIR between blackout screen and thermal screen [W m^{-2}]
    addAux(m, 'rBlScrThScr', fir(u.blScr, p.epsBlScrFir, p.epsThScrFir, u.thScr, x.tBlScr, x.tThScr, ...
        p.capBlScr, p.capThScr, p.simRes)); 
    
    % FIR between blackout screen and cover [W m^{-2}]
    addAux(m, 'rBlScrCovIn', fir(1, p.epsBlScrFir, m.a.epsCovFir, u.blScr.*m.a.tauThScrFirU, x.tBlScr, x.tCovIn, ...
        p.capBlScr, m.a.capCovIn, p.simRes));
    
    % FIR between blackout screen and sky [W m^{-2}]
    addAux(m, 'rBlScrSky', fir(1, p.epsBlScrFir, p.epsSky, m.a.tauCovFir.*u.blScr.*m.a.tauThScrFirU, x.tBlScr, d.tSky, ...
        p.capBlScr, p.capBlScr, p.simRes));

    % FIR between blackout screen and lamps [W m^{-2}]
    addAux(m, 'rLampBlScr', fir(p.aLamp, p.epsLampTop, p.epsBlScrFir, u.blScr, x.tLamp, x.tBlScr, ...
        p.capLamp, p.capBlScr, p.simRes));

	%% Natural ventilation - Section 9.7 [1]
    
	% Aperture of the roof [m^{2}]
	% Equation 67 [1]
    addAux(m, 'aRoofU', u.roof*p.aRoof);
    addAux(m, 'aRoofUMax', p.aRoof);
    addAux(m, 'aRoofMin', DynamicElement('0',0));
	
	% Aperture of the sidewall [m^{2}]
	% Equation 68 [1] 
	% (this is 0 in the Dutch greenhouse)
    addAux(m, 'aSideU', u.side*p.aSide); 
    
	% Ratio between roof vent area and total ventilation area [-]
	% (not very clear in the reference [1], but always 1 if m.a.aSideU == 0
	% so not important right now)
%     addAux(m, 'etaRoof', m.a.aRoofU./max(m.a.aRoofU+m.a.aSideU,0.01)); 
    addAux(m, 'etaRoof', '1'); 
    addAux(m, 'etaRoofNoSide', '1');
	
    % Ratio between side vent area and total ventilation area [-]
	% (not very clear in the reference [1], but always 0 if m.a.aSideU == 0
	% so not important right now)
    addAux(m, 'etaSide', '0'); 
	
    
	% Discharge coefficient [-]
	% Equation 73 [1]
    addAux(m, 'cD', p.cDgh*(1-p.etaShScrCd*u.shScr));
	
	% Discharge coefficient [-]
	% Equation 74 [-]
    addAux(m, 'cW', p.cWgh*(1-p.etaShScrCw*u.shScr));
    
	% Natural ventilation rate due to roof ventilation [m^{3} m^{-2} s^{-1}]
	% Equation 64 [1]
    addAux(m, 'fVentRoof2', u.roof*p.aRoof.*m.a.cD/(2.*p.aFlr).*...
        sqrt(abs(p.g*p.hVent*(x.tAir-d.tOut)./(2*(0.5*x.tAir+0.5*d.tOut+273.15))+m.a.cW.*d.wind.^2)));
    addAux(m, 'fVentRoof2Max', p.aRoof.*m.a.cD/(2.*p.aFlr).*...
        sqrt(abs(p.g*p.hVent*(x.tAir-d.tOut)./(2*(0.5*x.tAir+0.5*d.tOut+273.15))+m.a.cW.*d.wind.^2)));
    addAux(m, 'fVentRoof2Min', DynamicElement('0',0));
    
    
	% Ventilation rate through roof and side vents [m^{3} m^{-2} s^{-1}]
	% Equation 65 [1]
    addAux(m, 'fVentRoofSide2', m.a.cD/p.aFlr.*sqrt(...
        (m.a.aRoofU.*m.a.aSideU./sqrt(max(m.a.aRoofU.^2+m.a.aSideU.^2,0.01))).^2 .* ...
        (2*p.g*p.hSideRoof*(x.tAir-d.tOut)./(0.5*x.tAir+0.5*d.tOut+273.15))+...
        ((m.a.aRoofU+m.a.aSideU)/2).^2.*m.a.cW.*d.wind.^2));
    
	% Ventilation rate through sidewall only [m^{3} m^{-2} s^{-1}]
	% Equation 66 [1]
    addAux(m, 'fVentSide2', m.a.cD.*m.a.aSideU.*d.wind/(2*p.aFlr).*sqrt(m.a.cW));
    
	% Leakage ventilation [m^{3} m^{-2} s^{-1}]
	% Equation 70 [1]
    addAux(m, 'fLeakage', ifElse('d.wind<p.minWind',p.minWind*p.cLeakage,p.cLeakage*d.wind));
    
	% Total ventilation through the roof [m^{3} m^{-2} s^{-1}]
	% Equation 71 [1]
    addAux(m, 'fVentRoof', ifElse([m.a.etaRoof.def '>=p.etaRoofThr'],p.etaInsScr*m.a.fVentRoof2+p.leakTop*m.a.fLeakage,...
        p.etaInsScr*(max(u.thScr,u.blScr).*m.a.fVentRoof2+(1-max(u.thScr,u.blScr)).*m.a.fVentRoofSide2.*m.a.etaRoof)...
        +p.leakTop*m.a.fLeakage));
%     addAux(m, 'fVentRoofMin', ifElse([m.a.etaRoofNoSide.def '>=p.etaRoofThr'],p.etaInsScr*m.a.fVentRoof2Min+p.leakTop*m.a.fLeakage,...
%         p.leakTop*m.a.fLeakage));
%     addAux(m, 'fVentRoofMax', ifElse([m.a.etaRoofNoSide.def '>=p.etaRoofThr'],p.etaInsScr*m.a.fVentRoof2Max+p.leakTop*m.a.fLeakage,...
%         p.leakTop*m.a.fLeakage));
    
	% Total ventilation through side vents [m^{3} m^{-2} s^{-1}]
	% Equation 72 [1]
    addAux(m, 'fVentSide', ifElse([m.a.etaRoof.def '>=p.etaRoofThr'],p.etaInsScr*m.a.fVentSide2+(1-p.leakTop)*m.a.fLeakage,...
        p.etaInsScr*(max(u.thScr,u.blScr).*m.a.fVentSide2+(1-max(u.thScr,u.blScr)).*m.a.fVentRoofSide2.*m.a.etaSide)...
        +(1-p.leakTop)*m.a.fLeakage));
    
    %% Control rules 

    % hours since midnight [h]
    addAux(m, 'timeOfDay', 24*(x.time-floor(x.time))); 
    
    % control for the lamps [0/1]
    addAux(m, 'lampOn', sign(sign(p.lampsOffSun-d.iGlob)+1)... % lamps are off if sun is bright
        .*0.5.*(sign(m.a.timeOfDay-p.lampsOn)+sign(p.lampsOff-m.a.timeOfDay)));
        % lamps are on between p.lampsOn and p.lampsOff
    
    % 1 if lamps on, 0 if lamps off, with a smooth interpolation between
    addAux(m, 'smoothLamp', min(max(0,min(m.a.timeOfDay-p.lampsOn,1)),...
        max(0,min(1,p.lampsOff-m.a.timeOfDay))));
    
    % 1 if day, 0 if night. If lamps are on it is considered day
    addAux(m, 'isDay', max(m.a.smoothLamp,d.isDay));
    
    % Heating set point [C]
    addAux(m, 'heatSetPoint', m.a.isDay*p.tSpDay + (1-m.a.isDay)*p.tSpNight ...
        + p.heatCorrection*m.a.lampOn); % correction for LEDs when lamps are on
    
    % Ventilation due to excess heating set point [C]
    addAux(m, 'heatMax', m.a.heatSetPoint + p.heatDeadZone);    
    
    % CO2 set point [ppm]
    addAux(m, 'co2SetPoint', m.a.isDay*p.co2SpDay);
    
    % CO2 concentration in main compartment [ppm]
    addAux(m, 'co2InPpm', co2dens2ppm(x.tAir,1e-6*x.co2Air)); 
    
    % Ventilation due to excess heat [0-1, 0 means vents are closed]
    addAux(m, 'ventHeat', proportionalControl(x.tAir, m.a.heatMax, p.ventHeatPband, 0, 1));
    
    % Relative humidity [%]
    addAux(m, 'rhIn', 100*x.vpAir./satVp(x.tAir));
    
    % Ventilation due to excess humidity [0-1, 0 means vents are closed]
    addAux(m, 'ventRh', proportionalControl(m.a.rhIn, p.rhMax, p.ventRhPband, 0, 1));
    
    % Ventilation closure due to too cold temperatures [0-1, 0 means vents are closed]
    addAux(m, 'ventCold', proportionalControl(x.tAir, m.a.heatSetPoint-p.tVentOff, p.ventHeatPband, 0, 1));
    
    %% Convection and conduction - Section 5.3 [1]
    
    % density of air as it depends on pressure and temperature, see
    % https://en.wikipedia.org/wiki/Density_of_air
    addAux(m, 'rhoTop', p.mAir*p.pressure./((x.tTop+273.15)*p.R));
    addAux(m, 'rhoAir', p.mAir*p.pressure./((x.tAir+273.15)*p.R));
    
    % note a mistake in [1] - says rhoAirMean is the mean density "of the
    % greenhouse and the outdoor air". See [4], where rhoMean is "the mean
    % density of air beneath and above the screen".
    addAux(m, 'rhoAirMean', 0.5*(m.a.rhoTop+m.a.rhoAir));
    
	% Air flux through the thermal screen [m s^{-1}]
	% Equation 40 [1]
    % There is a mistake in [1], see equation 5.68, pg. 91, [4]
    % tOut, rhoOut, should be tTop, rhoTop
    % There is also a mistake in [4], whenever sqrt is taken, abs should be included
    addAux(m, 'fThScr', u.thScr*p.kThScr.*(abs((x.tAir-x.tTop)).^0.66) + ... 
        ((1-u.thScr)./m.a.rhoAirMean).*sqrt(0.5*m.a.rhoAirMean.*(1-u.thScr).*p.g.*abs(m.a.rhoAir-m.a.rhoTop)));
    
    % Air flux through the blackout screen [m s^{-1}]
    addAux(m, 'fBlScr', u.blScr*p.kBlScr.*(abs((x.tAir-x.tTop)).^0.66) + ... 
        ((1-u.blScr)./m.a.rhoAirMean).*sqrt(0.5*m.a.rhoAirMean.*(1-u.blScr).*p.g.*abs(m.a.rhoAir-m.a.rhoTop)));
    
    % Air flux through the screencs [m s^{-1}]
    addAux(m, 'fScr', min(m.a.fThScr,m.a.fBlScr));
        
    % Setpoint for closing the thermal screen [C]
    addAux(m, 'thScrSp', m.a.isDay.*p.thScrSpDay+(1-m.a.isDay).*p.thScrSpNight);
    
    % Control of the thermal screen based on heat loss requirements [0-1, 0 is fully open]
    addAux(m, 'thScrHeat', proportionalControl(d.tOut, m.a.thScrSp, p.thScrPband, 0, 1));
    
    % Control of thermal screen closure due to too low temperatures 
    % (even if rh is too high) [0-1, 0 is fully open]
    addAux(m, 'thScrCold', proportionalControl(d.tOut, m.a.thScrSp-p.tScreenOn, p.thScrPband, 0, 1));
    
    % Control of thermal screen due to high humidity [0-1, 0 is fully open]
    addAux(m, 'ventThScr', proportionalControl(m.a.rhIn, p.ventThScr, p.ventThScrPband, 1, 0));
    
	%% Convective and conductive heat fluxes [W m^{-2}]
	% Table 4 [1]
	% The variables for hec (heat exchange coefficient, [W m^{-2} K^{-1}])
	% are introduced to prevent overshoot when simulating
    
    % Forced ventilation (doesn't exist in current gh)
    addAux(m, 'fVentForced', DynamicElement('0', 0));
    
	% Between canopy and air in main compartment [W m^{-2}]
    addAux(m, 'hecCanAir', hec(2*p.alfaLeafAir*m.a.lai,p.capAir,m.a.capCan, p.simRes));
    addAux(m, 'hCanAir', sensible(m.a.hecCanAir, x.tCan, x.tAir));
	
	% Between air in main compartment and floor [W m^{-2}]
    addAux(m, 'hecAirFlr', hec(ifElse('x.tFlr>x.tAir',1.7*nthroot(abs(x.tFlr-x.tAir),3),1.3*nthroot(abs(x.tAir-x.tFlr),4)),...
        p.capAir, p.capFlr, p.simRes));
    addAux(m, 'hAirFlr', sensible(m.a.hecAirFlr, x.tAir,x.tFlr));
		
	% Between air in main compartment and thermal screen [W m^{-2}]
    addAux(m, 'hecAirThScr', hec(1.7.*u.thScr.*nthroot(abs(x.tAir-x.tThScr),3),...
        p.capAir, p.capThScr, p.simRes));
    addAux(m, 'hAirThScr', sensible(m.a.hecAirThScr, x.tAir,x.tThScr));
    
	% Between air in main compartment and blackout screen [W m^{-2}]
    addAux(m, 'hecAirBlScr', hec(1.7.*u.blScr.*nthroot(abs(x.tAir-x.tBlScr),3),...
        p.capAir, p.capBlScr, p.simRes));
    addAux(m, 'hAirBlScr', sensible(m.a.hecAirBlScr, x.tAir,x.tBlScr));
		
	% Between air in main compartment and outside air [W m^{-2}]
    addAux(m, 'hecAirOut', hec(p.rhoAir*p.cPAir*(m.a.fVentSide+m.a.fVentForced), ...
        p.capAir, p.capAir, p.simRes));
    addAux(m, 'hAirOut', sensible(m.a.hecAirOut, x.tAir, d.tOut));
		
	% Between air in main and top compartment [W m^{-2}]
    addAux(m, 'hecAirTop', hec(p.rhoAir*p.cPAir*m.a.fScr, p.capAir, p.capTop, p.simRes));
    addAux(m, 'hAirTop', sensible(m.a.hecAirTop, x.tAir, x.tTop));
	
	% Between thermal screen and top compartment [W m^{-2}]
    addAux(m, 'hecThScrTop', hec(1.7.*u.thScr.*nthroot(abs(x.tThScr-x.tTop),3),...
        p.capTop, p.capThScr, p.simRes));
    addAux(m, 'hThScrTop', sensible(m.a.hecThScrTop, x.tThScr,x.tTop));
    
    % Between blackout screen and top compartment [W m^{-2}]
    addAux(m, 'hecBlScrTop', hec(1.7.*u.blScr.*nthroot(abs(x.tBlScr-x.tTop),3),...
        p.capTop, p.capBlScr, p.simRes));
    addAux(m, 'hBlScrTop', sensible(m.a.hecBlScrTop, x.tBlScr,x.tTop));
		
	% Between top compartment and cover [W m^{-2}]
    addAux(m, 'hecTopCovIn', hec(p.cHecIn*nthroot(abs(x.tTop-x.tCovIn),3)*p.aCov/p.aFlr,...
        m.a.capCovIn,p.capTop,p.simRes));
    addAux(m, 'hTopCovIn', sensible(m.a.hecTopCovIn, x.tTop,x.tCovIn));
		
	% Between top compartment and outside air [W m^{-2}]
    addAux(m, 'hecTopOut', hec(p.rhoAir*p.cPAir*m.a.fVentRoof, p.capTop, p.capTop, p.simRes));
    addAux(m, 'hTopOut', sensible(m.a.hecTopOut, x.tTop, d.tOut));
	
	% Between cover and outside air [W m^{-2}]
    addAux(m, 'hecCovEOut', hec(p.aCov/p.aFlr*(p.cHecOut1+p.cHecOut2*d.wind.^p.cHecOut3), ...
        m.a.capCovE, m.a.capCovE, p.simRes));
    addAux(m, 'hCovEOut', sensible(m.a.hecCovEOut, x.tCovE, d.tOut));
	
	% Between pipes and air in main compartment [W m^{-2}]
    addAux(m, 'hecPipeAir', hec(1.99*pi*p.phiPipeE*p.lPipe*(abs(x.tPipe-x.tAir)).^0.32, ...
        p.capAir, p.capPipe, p.simRes));
    addAux(m, 'hPipeAir', sensible(m.a.hecPipeAir, x.tPipe, x.tAir));
		
	% Between floor and soil layer 1 [W m^{-2}]
    addAux(m, 'hecFlrSo1', hec(2/(p.hFlr/p.lambdaFlr+p.hSo1/p.lambdaSo),...
        p.capFlr, p.capSo1, p.simRes));
    addAux(m, 'hFlrSo1', sensible(m.a.hecFlrSo1, x.tFlr, x.tSo1));
	
	% Between soil layers 1 and 2 [W m^{-2}]
    addAux(m, 'hecSo1So2', hec(2*p.lambdaSo/(p.hSo1+p.hSo2), p.capSo1, p.capSo2, p.simRes));
    addAux(m, 'hSo1So2', sensible(m.a.hecSo1So2, x.tSo1, x.tSo2));
    
	% Between soil layers 2 and 3 [W m^{-2}]
	addAux(m, 'hecSo2So3', hec(2*p.lambdaSo/(p.hSo2+p.hSo3), p.capSo2, p.capSo3, p.simRes));
    addAux(m, 'hSo2So3', sensible(m.a.hecSo2So3, x.tSo2, x.tSo3));
    
	% Between soil layers 3 and 4 [W m^{-2}]
	addAux(m, 'hecSo3So4', hec(2*p.lambdaSo/(p.hSo3+p.hSo4), p.capSo3, p.capSo4, p.simRes));
    addAux(m, 'hSo3So4', sensible(m.a.hecSo3So4, x.tSo3, x.tSo4));
    
	% Between soil layers 4 and 5 [W m^{-2}]
	addAux(m, 'hecSo4So5', hec(2*p.lambdaSo/(p.hSo4+p.hSo5), p.capSo4, p.capSo5, p.simRes));
    addAux(m, 'hSo4So5', sensible(m.a.hecSo4So5, x.tSo4, x.tSo5));
    
    % Between soil layer 5 and the external soil temperature [W m^{-2}]
    % See Equations 4 and 77 [1]
    addAux(m, 'hecSo5SoOut', hec(2*p.lambdaSo/(p.hSo5+p.hSoOut), p.capSo5, p.capSo5, p.simRes));
    addAux(m, 'hSo5SoOut', sensible(m.a.hecSo5SoOut, x.tSo5, d.tSoOut));
    
    % Conductive heat flux through the lumped cover [W K^{-1} m^{-2}]
    % See comment after Equation 18 [1]
    addAux(m, 'hecCovInCovE', hec(1./(p.hRf/p.lambdaRf+u.shScrPer*p.hShScrPer/p.lambdaShScrPer),...
        m.a.capCovE, m.a.capCovIn, p.simRes));
    addAux(m, 'hCovInCovE', sensible(m.a.hecCovInCovE, x.tCovIn, x.tCovE));

    % Between lamps and air in main compartment [W m^{-2}]
%     if strcmp(lamp, 'led') % LEDs are treated as heating pipes
%         addAux(m, 'hecLampAir', hec(1.99*pi*p.phiGroPipeE*p.lGroPipe*nthroot(abs(x.tLamp-x.tAir),3), ...
%             p.capAir, p.capLamp, p.simRes));
%     else
        addAux(m, 'hecLampAir', hec(p.hecLampAir, p.capAir, p.capLamp, p.simRes));
%     end
    
    addAux(m, 'hLampAir', sensible(m.a.hecLampAir, x.tLamp, x.tAir));
    
    % Between grow pipes and air in main compartment [W m^{-2}]
    addAux(m, 'hecGroPipeAir', hec(1.99*pi*p.phiGroPipeE*p.lGroPipe*(abs(x.tGroPipe-x.tAir)).^0.32, ...
        p.capAir, p.capGroPipe, p.simRes));
    addAux(m, 'hGroPipeAir', sensible(m.a.hecGroPipeAir, x.tGroPipe, x.tAir));
        
   % Between interlights and air in main compartment [W m^{-2}]
%     addAux(m, 'hecIntLampAir', hec(1.99*pi*p.phiGroPipeE*p.lGroPipe*nthroot(abs(x.tIntLamp-x.tAir),3), ...
%             p.capAir, p.capIntLamp, p.simRes));

    addAux(m, 'hecIntLampAir', hec(p.hecIntLampAir, p.capAir, p.capIntLamp, p.simRes));
    addAux(m, 'hIntLampAir', sensible(m.a.hecIntLampAir, x.tIntLamp, x.tAir));
    
    % Radiative heat from the interlights absorbed by the air
    addAux(m, 'rIntLampAir', '0');

    
    %% Canopy transpiration - Section 8 [1]
    
	% Smooth switch between day and night [-]
	% Equation 50 [1]
    addAux(m, 'sRs', 1./(1+exp(p.sRs.*(m.a.rCan-p.rCanSp))));
        
	% Parameter for co2 influence on stomatal resistance [ppm{CO2}^{-2}]
	% Equation 51 [1]
    addAux(m, 'cEvap3', p.cEvap3Night*(1-m.a.sRs)+p.cEvap3Day*m.a.sRs);
		
	% Parameter for vapor pressure influence on stomatal resistance [Pa^{-2}]
    addAux(m, 'cEvap4', p.cEvap4Night*(1-m.a.sRs)+p.cEvap4Day*m.a.sRs);
    
	% Radiation influence on stomatal resistance [-]
	% Equation 49 [1]
    addAux(m, 'rfRCan', (m.a.rCan+p.cEvap1)./(m.a.rCan+p.cEvap2));
    
	% CO2 influence on stomatal resistance [-]
	% Equation 49 [1]
    addAux(m, 'rfCo2', min(1.5,1+m.a.cEvap3.*(p.etaMgPpm*x.co2Air-200).^2));
        % perhpas replace p.etaMgPpm*x.co2Air with a.co2InPpm
    
	% Vapor pressure influence on stomatal resistance [-]
	% Equation 49 [1]
    addAux(m, 'rfVp', min(5.8, 1+m.a.cEvap4.*(satVp(x.tCan)-x.vpAir).^2));
    
	% Stomatal resistance [s m^{-1}]
	% Equation 48 [1]
    addAux(m, 'rS', p.rSMin.*m.a.rfRCan.*m.a.rfCo2.*m.a.rfVp);
    
	% Vapor transfer coefficient of canopy transpiration [kg m^{-2} Pa^{-1} s^{-1}]
	% Equation 47 [1]
    addAux(m, 'vecCanAir', 2*p.rhoAir*p.cPAir*m.a.lai./...
        (p.L*p.gamma*(p.rB+m.a.rS)));
    
	% Canopy transpiration [kg m^{-2} s^{-1}]
	% Equation 46 [1]
    addAux(m, 'mvCanAir', (satVp(x.tCan)-x.vpAir).*m.a.vecCanAir); 
        
	%% Vapor fluxes  - Section 6 [1]
	
	% Vapor fluxes currently not included in the model - set at 0
    addAux(m, 'mvPadAir', DynamicElement('0', 0));
    addAux(m, 'mvFogAir', DynamicElement('0', 0));
    addAux(m, 'mvBlowAir', DynamicElement('0', 0));
    addAux(m, 'mvAirOutPad', DynamicElement('0', 0));
    addAux(m, 'mvAirMech', DynamicElement('0', 0));
    
    
    % Condensation from main compartment on thermal screen [kg m^{-2} s^{-1}]
	% Table 4 [1], Equation 42 [1]
    addAux(m, 'mvAirThScr', cond(1.7*u.thScr.*nthroot(abs(x.tAir-x.tThScr),3), ...
        x.vpAir,satVp(x.tThScr), m.a.capVpAir, p.simRes));
    
    % Condensation from main compartment on blackout screen [kg m^{-2} s^{-1}]
    addAux(m, 'mvAirBlScr', cond(1.7*u.blScr.*nthroot(abs(x.tAir-x.tBlScr),3), ...
        x.vpAir,satVp(x.tBlScr), m.a.capVpAir, p.simRes));
    
    % Condensation from top compartment to cover [kg m^{-2} s^{-1}]
	% Table 4 [1]
    addAux(m, 'mvTopCovIn', cond(p.cHecIn*nthroot(abs(x.tTop-x.tCovIn),3)*p.aCov/p.aFlr,...
        x.vpTop,satVp(x.tCovIn), m.a.capVpTop, p.simRes));
    
	% Vapor flux from main to top compartment [kg m^{-2} s^{-1}]
    addAux(m, 'mvAirTop', airMv(m.a.fScr, x.vpAir, x.vpTop, x.tAir, x.tTop, ...
        m.a.capVpAir, m.a.capVpTop, p.simRes));
	
	% Vapor flux from top compartment to outside [kg  m^{-2} s^{-1}]
    addAux(m, 'mvTopOut', airMv(m.a.fVentRoof, x.vpTop, d.vpOut, x.tTop, d.tOut, ...
        m.a.capVpTop, m.a.capVpTop, p.simRes));
	
	% Vapor flux from main compartment to outside [kg m^{-2} s^{-1}]
    addAux(m, 'mvAirOut', airMv(m.a.fVentSide+m.a.fVentForced, x.vpAir, d.vpOut, x.tAir, d.tOut, ...
        m.a.capVpAir, m.a.capVpAir, p.simRes));
    
    %% Latent heat fluxes - Section 5.4 [1]
	% Equation 41 [1]
	
	% Latent heat flux by transpiration [W m^{-2}]
    addAux(m, 'lCanAir', p.L*m.a.mvCanAir);
	
	% Latent heat flux by condensation on thermal screen [W m^{-2}]
    addAux(m, 'lAirThScr', p.L*m.a.mvAirThScr);
    
    % Latent heat flux by condensation on blackout screen [W m^{-2}]
    addAux(m, 'lAirBlScr', p.L*m.a.mvAirBlScr);
	
	% Latent heat flux by condensation on cover [W m^{-2}]
    addAux(m, 'lTopCovIn', p.L*m.a.mvTopCovIn);
    
    %% Canopy photosynthesis - Section 4.1 [2]
	
	% PAR absorbed by the canopy [umol{photons} m^{-2} s^{-1}]
	% Equation 17 [2]
    addAux(m, 'parCan', p.parJtoUmolLamp*m.a.rParLampCan + p.parJtoUmolSun*m.a.rParSunCan + ...
        p.parJtoUmolIntLamp*m.a.rParIntLampCan);
    
	% Maximum rate of electron transport rate at 25C [umol{e-} m^{-2} s^{-1}]
	% Equation 16 [2]
    addAux(m, 'j25CanMax', m.a.lai*p.j25LeafMax);
    
	% CO2 compensation point [ppm]
	% Equation 23 [2]
    addAux(m, 'gamma', divNoBracks(p.j25LeafMax,(m.a.j25CanMax)*1).*p.cGamma.*x.tCan + ...
        20*p.cGamma.*(1-divNoBracks(p.j25LeafMax,(m.a.j25CanMax)*1)));
    
	% CO2 concentration in the stomata [ppm]
	% Equation 21 [2]
    addAux(m, 'co2Stom', p.etaCo2AirStom*m.a.co2InPpm);
    
	% Potential rate of electron transport [umol{e-} m^{-2} s^{-1}]
	% Equation 15 [2]
	% Note that R in [2] is 8.314 and R in [1] is 8314
    addAux(m, 'jPot', m.a.j25CanMax.*exp(p.eJ*(x.tCan+273.15-p.t25k)./(1e-3*p.R*(x.tCan+273.15)*p.t25k)).*...
        (1+exp((p.S*p.t25k-p.H)./(1e-3*p.R*p.t25k)))./...
        (1+exp((p.S*(x.tCan+273.15)-p.H)./(1e-3*p.R*(x.tCan+273.15)))));
    
	% Electron transport rate [umol{e-} m^{-2} s^{-1}]
	% Equation 14 [2]
    addAux(m, 'j', (1/(2*p.theta))*(m.a.jPot+p.alpha*m.a.parCan-...
        sqrt((m.a.jPot+p.alpha*m.a.parCan).^2-4*p.theta*m.a.jPot.*p.alpha.*m.a.parCan)));
    
	% Photosynthesis rate at canopy level [umol{co2} m^{-2} s^{-1}]
	% Equation 12 [2]
    addAux(m, 'p', m.a.j.*(m.a.co2Stom-m.a.gamma)./(4*(m.a.co2Stom+2*m.a.gamma)));
	
	% Photrespiration [umol{co2} m^{-2} s^{-1}]
	% Equation 13 [2]
    addAux(m, 'r', m.a.p.*m.a.gamma./m.a.co2Stom);
    
	% Inhibition due to full carbohydrates buffer [-]
    % Equation 11, Equation B.1, Table 5 [2]
    addAux(m, 'hAirBuf', 1./(1+exp(5e-4*(x.cBuf-p.cBufMax))));
    
	% Net photosynthesis [mg{CH2O} m^{-2} s^{-1}]
	% Equation 10 [2]
    addAux(m, 'mcAirBuf', p.mCh2o*m.a.hAirBuf.*(m.a.p-m.a.r));
    
    %% Carbohydrate buffer
    % Temperature effect on structural carbon flow to organs
    % Equation 28 [2]
    addAux(m, 'gTCan24', 0.047*x.tCan24+0.06);
    
    % Inhibition of carbohydrate flow to the organs
    % Equation B.3 [2]
    addAux(m, 'hTCan24', 1./(1+exp(-1.1587*(x.tCan24-p.tCan24Min))).* ...
        1./(1+exp(1.3904*(x.tCan24-p.tCan24Max))));
    
    % Inhibition of carbohydrate flow to the fruit
    % Equation B.3 [2]
    addAux(m, 'hTCan', 1./(1+exp(-0.869*(x.tCan-p.tCanMin))).* ...
        1./(1+exp(0.5793*(x.tCan-p.tCanMax))));
    
    % Inhibition due to development stage 
    % Equation B.6 [2]
    addAux(m, 'hTCanSum', 0.5*(x.tCanSum/p.tEndSum+...
        sqrt((x.tCanSum./p.tEndSum).^2+1e-4)) - ...
        0.5*((x.tCanSum-p.tEndSum)./p.tEndSum+...
        sqrt(((x.tCanSum-p.tEndSum)/p.tEndSum).^2 + 1e-4)));
    
    % Inhibition due to insufficient carbohydrates in the buffer [-]
    % Equation 26 [2]
    addAux(m, 'hBufOrg', 1./(1+exp(-5e-3*(x.cBuf-p.cBufMin))));
    
    % Carboyhdrate flow from buffer to leaves [mg{CH2O} m^{2} s^{-1}]
    % Equation 25 [2]
    addAux(m, 'mcBufLeaf', m.a.hBufOrg.*m.a.hTCan24.*m.a.gTCan24.*m.p.rgLeaf);
    
    % Carboyhdrate flow from buffer to stem [mg{CH2O} m^{2} s^{-1}]
    % Equation 25 [2]
    addAux(m, 'mcBufStem', m.a.hBufOrg.*m.a.hTCan24.*m.a.gTCan24.*m.p.rgStem);
    
    % Carboyhdrate flow from buffer to fruit [mg{CH2O} m^{2} s^{-1}]
    % Equation 24 [2]
    addAux(m, 'mcBufFruit', m.a.hBufOrg.*...
        m.a.hTCan.*m.a.hTCan24.*m.a.hTCanSum.*m.a.gTCan24.*m.p.rgFruit);
    
    %% Growth and maintenance respiration - Section 4.4 [2]
	
	% Growth respiration [mg{CH2O} m^{-2] s^{-1}]
	% Equations 43-44 [2]
    addAux(m, 'mcBufAir', p.cLeafG*m.a.mcBufLeaf + p.cStemG*m.a.mcBufStem ...
        +p.cFruitG*m.a.mcBufFruit);
    
	% Leaf maintenance respiration [mg{CH2O} m^{-2} s^{-1}]
	% Equation 45 [2]
    addAux(m, 'mcLeafAir', (1-exp(-p.cRgr*p.rgr)).*p.q10m.^(0.1*(x.tCan24-25)).* ...
        x.cLeaf*p.cLeafM);
    
    % Stem maintenance respiration [mg{CH2O} m^{-2} s^{-1}]
	% Equation 45 [2]
    addAux(m, 'mcStemAir', (1-exp(-p.cRgr*p.rgr)).*p.q10m.^(0.1*(x.tCan24-25)).* ...
        x.cStem*p.cStemM);
    
    % Fruit maintenance respiration [mg{CH2O} m^{-2} s^{-1}]
	% Equation 45 [2]
    addAux(m, 'mcFruitAir', (1-exp(-p.cRgr*p.rgr)).*p.q10m.^(0.1*(x.tCan24-25)).* ...
        x.cFruit*p.cFruitM);
    
    % Total maintenance respiration [mg{CH2O} m^{-2} s^{-1}]
	% Equation 45 [2]
    addAux(m, 'mcOrgAir', m.a.mcLeafAir+m.a.mcStemAir+m.a.mcFruitAir);
    
    %% Leaf pruning and fruit harvest
    % Leaf pruning [mg{CH2O} m^{-2] s^{-1}]
    % Equation B.5
    addAux(m, 'mcLeafHar', ifElse('x.cLeaf<p.cLeafMax',0, x.cLeaf-p.cLeafMax));
    
    % Fruit harvest [mg{CH2O} m^{-2] s^{-1}]
    addAux(m, 'mcFruitHar', ifElse('x.cFruit<p.cFruitMax',0,x.cFruit-p.cFruitMax));
    
	%% CO2 Fluxes - Section 7 [1]
	
    % Net crop assimilation [mg{CO2} m^{-2} s^{-1}]
    % It is assumed that for every mol of CH2O in net assimilation, a mol
    % of CO2 is taken from the air, thus the conversion uses molar masses
    addAux(m, 'mcAirCan', (p.mCo2/p.mCh2o)*(m.a.mcAirBuf-m.a.mcBufAir-m.a.mcOrgAir));
    
    % Other CO2 flows [mg{CO2} m^{-2} s^{-1}]
	% Equation 45 [1]
	
	% From main to top compartment 
    addAux(m, 'mcAirTop', airMc(m.a.fScr, x.co2Air, x.co2Top, p.capCo2Air, p.capCo2Top, p.simRes));
	
	% From top compartment outside
    addAux(m, 'mcTopOut', airMc(m.a.fVentRoof, x.co2Top, d.co2Out, p.capCo2Top, p.capCo2Top, p.simRes));
    
	% From main compartment outside
	addAux(m, 'mcAirOut', airMc(m.a.fVentSide+m.a.fVentForced, x.co2Air, d.co2Out, p.capCo2Air, p.capCo2Air, p.simRes));
    
    %% Heat from boiler - Section 9.2 [1]
	
	% Heat from boiler to pipe rails [W m^{-2}]
	% Equation 55 [1]
    addAux(m, 'hBoilPipe', u.boil*p.pBoil/p.aFlr);
    
    % Heat from boiler to grow pipes [W m^{-2}]
    addAux(m, 'hBoilGroPipe', u.groBoil*p.pGroBoil/p.aFlr);
    
    %% External CO2 source - Section 9.9 [1]
	
	% CO2 injection [mg m^{-2} s^{-1}]
	% Equation 76 [1]
    addAux(m, 'mcExtAir', u.extCo2*p.phiExtCo2/p.aFlr);
    
    %% Objects not currently included in the model
    addAux(m, 'mcBlowAir', DynamicElement('0',0));
    addAux(m, 'mcPadAir', DynamicElement('0',0));
    addAux(m, 'hPadAir', DynamicElement('0',0));
    addAux(m, 'hPasAir', DynamicElement('0',0));
    addAux(m, 'hBlowAir', DynamicElement('0',0));
    addAux(m, 'hAirPadOut', DynamicElement('0',0));
    addAux(m, 'hMechAir', DynamicElement('0',0));
    addAux(m, 'hAirOutPad', DynamicElement('0',0));
    addAux(m, 'lAirFog', DynamicElement('0',0));
    addAux(m, 'hIndPipe', DynamicElement('0',0));
    addAux(m, 'hGeoPipe', DynamicElement('0',0));
    
    %% Lamp cooling
    addAux(m, 'lampCool', p.lampCool*m.a.lampIn);
    
    
end

function sat = satVp(temp)
% saturated vapor pressure (Pa) at temperature temp (C)
% Calculation based on 
%   http://www.conservationphysics.org/atmcalc/atmoclc2.pdf

    % parameters used in the conversion
    p = [610.78 238.3 17.2694 -6140.4 273 28.916];
        % default value is [610.78 238.3 17.2694 -6140.4 273 28.916]
    
    sat = p(1)*exp(p(3)*temp./(temp+p(2))); 
        % Saturation vapor pressure of air in given temperature [Pa]
end

function de = tau12(tau1, tau2, ~, rho1Dn, rho2Up, ~)
% Transmission coefficient of a double layer [-]
% Equation 14 [1]
    de = tau1.*tau2./(1-rho1Dn.*rho2Up);    
end

function de = rhoUp(tau1, ~, rho1Up, rho1Dn, rho2Up, ~)
% Reflection coefficient of the top layer of a double layer [-]
% Equation 15 [1]
    de = rho1Up + (tau1.^2.*rho2Up)./(1-rho1Dn.*rho2Up);
end

function de = rhoDn(~, tau2, ~, rho1Dn, rho2Up, rho2Dn)
% Reflection coefficient of the top layer of a double layer [-]
% Equation 15 [1]
    de = rho2Dn + (tau2.^2.*rho1Dn)./(1-rho1Dn.*rho2Up);
end

function de = fir(a1,eps1,eps2,f12,t1,t2, cap1, cap2, simRes)
% Net far infrared flux from 1 to 2 [W m^{-2}]
% Equation 37 [1]
    
    sigma = 5.67e-8;
    kelvin = 273.15;
    
    % avoid overshoot, nice idea but for some reason makes the simulation
    % very noisy
%     
    aef = a1.*eps1.*eps2.*f12*sigma;
    tcr = (t1-t2).*cap1./(simRes*( (t1+kelvin).^4 - (t2+kelvin).^4 ));
    
    de = ifElse([aef.def '<=' tcr.def], ...
        a1.*eps1.*eps2.*f12*sigma.*( (t1+kelvin).^4 - (t2+kelvin).^4 ), ...
        (t1-t2).*cap2./simRes);

    % alternatively calculate directly without considering overshoot
    de = a1.*eps1.*eps2.*f12*sigma.*...
         ( (t1+kelvin).^4 - (t2+kelvin).^4 );
end

function de = sensible(hec, t1, t2)
% Sensible heat flux (convection or conduction) [W m^{-2}]
% Equation 39 [1]

    de = hec.*(t1-t2);
end

function de = cond(hec, vp1, vp2, cap, simRes)
% Vapor flux from the air to an object [kg m^{-2} s^{-1}]
% Equation 42 [1]

    de = max(0, min(6.4e-9*hec.*(vp1-vp2),cap./simRes.*(vp1-vp2)));
    
    de = max(0, 6.4e-9*hec.*(vp1-vp2));
end

function de = airMv(f12, vp1, vp2, t1, t2, cap1, cap2, simRes)
% Vapor flux accompanying an air flux [kg m^{-2} s^{-1}]
% Equation 44 [1]

    mWater = 18;
    r = 8.314e3;
	kelvin = 273.15;
    
    % vp1 is bigger than vp2
    de1 = min((mWater/r)*abs(f12).*(vp1./(t1+kelvin)-vp2./(t2+kelvin)),...
        min((vp1-vp2).*cap1./simRes,(vp1-vp2).*cap2./simRes));
    
    % vp2 is bigger
    de2 = max((mWater/r)*abs(f12).*(vp1./(t1+kelvin)-vp2./(t2+kelvin)),...
        max((vp1-vp2).*cap1./simRes,(vp1-vp2).*cap2./simRes));
    
    de = ifElse([vp1.def '>' vp2.def], de1, de2);
    
    de = (mWater/r)*abs(f12).*(vp1./(t1+kelvin)-vp2./(t2+kelvin));
end

function de = airMc(f12, c1, c2, cap1, cap2, simRes)
% Co2 flux accompanying an air flux [kg m^{-2} s^{-1}]
% Equation 45 [1]
    de = min(abs(f12), min(cap1./simRes, cap2./simRes)).*(c1-c2);
    
    de = abs(f12).*(c1-c2);
end

function de = hec(h,cap1,cap2,simRes)
    de = min(abs(h),min(abs(cap1/simRes),abs(cap2/simRes)));
    
    de = abs(h);
end

% function de = sign(deIn)
% % replace the discontinous sign function with something smooth
%     de = 2./(1+exp(-deIn))-1;
% end